import bpy


class ConnectionPanel(bpy.types.Panel):
    bl_idname = 'rebocap_pt_connection_panel'
    bl_label = 'Connection'
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "REBOCAP CONNECTION"

    # @classmethod
    # def poll(cls, ctx):
    #     return ctx.active_object != None and ctx.active_object.type == 'ARMATURE'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.bones = [
            "Pelvis", "L_UpLeg", "R_UpLeg", "Spine", "L_DownLeg", "R_DownLeg",
            "Chest", "L_Foot", "R_Foot", "UpChest", "L_Toe", "R_Toe",
            "Neck", "L_Shoulder", "R_Shoulder", "Head", "L_UpArm", "R_UpArm",
            "L_DownArm", "R_DownArm", "L_Palm", "R_Palm", "L_Fingers", "R_Fingers",
        ]
        self.bones_map = ['' for _ in range(24)]
        self.center_bone = [0, 3, 6, 9, 12, 15]
        self.left_bone = [1, 4, 7, 10, 13, 16, 18, 20]
        self.right_bone = [2, 5, 8, 11, 14, 17, 19, 21]
        self.side_names = [self.bones[i][2:] for i in self.left_bone]

    def draw_direct(self, ctx):
        pass

    def draw(self, ctx):
        layout = self.layout
        rebocap_bone_map = ctx.scene.rebocap_bone_map
        col = layout.column()
        row = col.row(align=True)
        row.label(text='Port')
        row.prop(ctx.scene, 'port', text='')
        row = col.row(align=True)
        if ctx.scene.open is False:
            row.operator('rebocap.connect', icon='URL')
        else:
            row.operator('rebocap.disconnect', icon='URL')
        if ctx.scene.open is True:
            row = col.row(align=True)
            if not ctx.scene.recording:
                row.operator('rebocap.start_record', icon='REC')
            else:
                row.operator('rebocap.stop_record', icon='REC', depress=True)
        if ctx.active_object is None or ctx.active_object.type != 'ARMATURE':
            return
        row = col.row(align=True)
        if ctx.scene.recording:
            row.label(text='Drive Type')
            row.label(text=ctx.active_object.rebocap_drive_type)
        else:
            row.label(text='Drive Type')
            row.prop(ctx.active_object, 'rebocap_drive_type', text='')
            row = col.row(align=True)
            row.label(text='Source')
            row.prop_search(ctx.active_object, 'rebocap_source_armature', ctx.scene, 'objects', text='')
            
        if ctx.active_object.rebocap_drive_type == 'DIRECT':
            self.draw_direct(ctx)
        else:
            source = bpy.data.objects.get(ctx.active_object.rebocap_source_armature)
            if source and source.type == 'ARMATURE':
                row = col.row()
                row.operator('rebocap.auto_map_bone', text='Auto Detect', icon='BONE_DATA')
                col = layout.column()
                all_bind = True
                has_bind = False

                box = layout.box()
                box.label(text="VRM Required Bones", icon="ARMATURE_DATA")

                row = box.row(align=True).split(factor=0.15, align=True)
                column0 = row.column(align=True)
                column1 = row.column(align=True)
                for i in range(len(self.center_bone)):
                    name = self.bones[self.center_bone[i]]
                    column0.label(text=name)
                    new_row = column1.split(factor=0.93, align=True)
                    new_row.prop_search(rebocap_bone_map, f'node_{self.center_bone[i]}', source.pose, 'bones', text='')
                    new_row.operator("object.pick_bone", text="", icon='EYEDROPPER').bone_type = f'node_{self.center_bone[i]}'

                row = box.row(align=True).split(factor=0.15, align=True)
                column0 = row.column(align=True)
                column1 = row.column(align=True)
                column2 = row.column(align=True)
                column0.label(text='')
                column1.label(text='Left')
                column2.label(text='Right')
                for i in range(len(self.side_names)):
                    column0.label(text=self.side_names[i])
                    new_row1 = column1.split(factor=0.93, align=True)
                    new_row1.prop_search(rebocap_bone_map,  f'node_{self.left_bone[i]}', source.pose, 'bones', text='')
                    new_row1.operator("object.pick_bone", text="", icon='EYEDROPPER').bone_type = f'node_{self.left_bone[i]}'
                    # column1.prop_search(rebocap_bone_map, f'node_{self.left_bone[i]}', source.pose, 'bones', text='')

                    new_row2 = column2.split(factor=0.93, align=True)
                    new_row2.prop_search(rebocap_bone_map,  f'node_{self.right_bone[i]}', source.pose, 'bones', text='')
                    new_row2.operator("object.pick_bone", text="", icon='EYEDROPPER').bone_type = f'node_{self.right_bone[i]}'
                    # column2.prop_search(rebocap_bone_map, f'node_{self.right_bone[i]}', source.pose, 'bones', text='')
                    if i == 3:
                        column0.separator()
                        column1.separator()
                        column2.separator()

                box = layout.box()
                box.label(text="Foot Contact Positions", icon="ARMATURE_DATA")
                row = box.row(align=True).split(factor=0.15, align=True)
                column0 = row.column(align=True)
                column1 = row.column(align=True)
                column2 = row.column(align=True)
                column0.label(text='')
                column1.label(text='Left')
                column2.label(text='Right')

                label_names = ['toeLeft', 'toeCenter', 'toeRight', 'heelLeft', 'heelCenter', 'heelRight']
                for i in range(6):
                    column0.label(text=label_names[i])
                    column1.prop(rebocap_bone_map, f'foot_idx_{i}', text='')
                    column2.prop(rebocap_bone_map, f'foot_idx_{i + 6}', text='')
                    if i == 2:
                        column0.separator()
                        column1.separator()
                        column2.separator()

                # check bind
                check_attr = ['node_0', 'node_1', 'node_2', 'node_4', 'node_5', 'node_7', 'node_8']
                all_bind = True
                for attr in check_attr:
                    if getattr(rebocap_bone_map, attr, '') == '':
                        all_bind = False
                if all_bind:
                    row = col.row()
                    row.operator('rebocap.save_bone', text='Save Bone', icon='BONE_DATA')


class PickBoneOperator(bpy.types.Operator):
    bl_idname = "object.pick_bone"
    bl_label = "Pick Bone"

    bone_type: bpy.props.StringProperty()

    def execute(self, context):
        armature = context.object

        if armature and armature.type == 'ARMATURE':
            bpy.ops.object.mode_set(mode='POSE')
            selected_bones = [b.name for b in armature.data.bones if b.select]

            if selected_bones:
                setattr(context.scene.rebocap_bone_map, self.bone_type, selected_bones[0])
                self.report({'INFO'}, f"Selected bone: {selected_bones[0]}")
            else:
                self.report({'WARNING'}, "No bone selected.")
            return {'FINISHED'}
        else:
            self.report({'WARNING'}, "Please select an armature object.")
            return {'CANCELLED'}
