#include "rebocap_source.h"

#include "Roles/LiveLinkAnimationRole.h"
#include "Roles/LiveLinkAnimationTypes.h"
#include "rebocap_skeleton_data.h"

FRebocapSource::FRebocapSource(uint16_t port) {
  client_ = nullptr;
  port_ = port;
  rebocap_sdk_ = MakeShared<rebocap::RebocapWsSdk>(CoordinateSpaceType::UECoordinate, true);
  rebocap_sdk_->SetPoseMsgCallback([this](const QuatMsg* msg, rebocap::RebocapWsSdk* h) { pose_msg_callback(msg); });
  rebocap_sdk_->SetExceptionCloseCallback([this](rebocap::RebocapWsSdk* h) { status_ = false; });
  FString ThreadName(FString::Printf(TEXT("FRebocapSource%ld"), (long)(FDateTime::UtcNow().ToUnixTimestamp())));
  thread_ = FRunnableThread::Create(this, *ThreadName, 512 * 1024, TPri_Normal);

  UE_LOG(LogTemp, Display, TEXT("new rebocap source for port:%d"), port);
}

FRebocapSource::FRebocapSource(const FString& IPAddress, uint16_t port) {
  client_ = nullptr;
  port_ = port;
  rebocap_sdk_ = MakeShared<rebocap::RebocapWsSdk>(CoordinateSpaceType::UECoordinate, true);
  rebocap_sdk_->SetPoseMsgCallback([this](const QuatMsg* msg, rebocap::RebocapWsSdk* h) { pose_msg_callback(msg); });
  rebocap_sdk_->SetExceptionCloseCallback([this](rebocap::RebocapWsSdk* h) { status_ = false; });
  FString ThreadName(FString::Printf(TEXT("FRebocapSource%ld"), (long)(FDateTime::UtcNow().ToUnixTimestamp())));
  thread_ = FRunnableThread::Create(this, *ThreadName, 512 * 1024, TPri_Normal);

  UE_LOG(LogTemp, Display, TEXT("new rebocap source for port:%d"), port);
}

FRebocapSource::~FRebocapSource() {
  UE_LOG(LogTemp, Display, TEXT("Destroying Rebocap Source!!!"));
  ManualStop();
  clear_all_subjects();
  if (thread_ != nullptr) {
    thread_->Kill(true);
    delete thread_;
  }
}

void FRebocapSource::ReceiveClient(ILiveLinkClient* InClient, FGuid InSourceGuid) {
  client_ = InClient;
  source_guid_ = InSourceGuid;
}

void FRebocapSource::InitializeSettings(ULiveLinkSourceSettings* Settings) {
  saved_source_settings_ = Cast<URebocapSourceSettings>(Settings);
}

bool FRebocapSource::IsSourceStillValid() const { return client_ != nullptr; }

bool FRebocapSource::RequestSourceShutdown() {
  clear_all_subjects();
  return true;
}

void FRebocapSource::clear_all_subjects() {
  if (client_ == nullptr) {
    UE_LOG(LogTemp, Warning, TEXT("Client was null!!!!!!"));
    return;
  }
  for (auto& subject_name : subject_names_) {
    client_->RemoveSubject_AnyThread(FLiveLinkSubjectKey(source_guid_, subject_name));
  }
}

bool FRebocapSource::open() {
  if (status_) {
    return status_;
  }
  int ret = rebocap_sdk_->Open(port_, "reborn_app", 114514);
  status_ = ret == 0;
  UE_LOG(LogTemp, Warning, TEXT("open ws client status: %d!!"), ret);
  return status_;
}

uint32 FRebocapSource::Run() {
  uint16_t tick = 0;
  while (running_) {
    if (tick++ % 4 == 0) open();
    FPlatformProcess::Sleep(0.03);
  }
  rebocap_sdk_->Close();
  status_ = false;
  return 0;
}

void FRebocapSource::ManualStop() {
  running_ = false;
}

bool FRebocapSource::ManualStart(uint16_t port) {
  port_ = port;
  if (!running_ && !status_) {
    running_ = true;
    FString ThreadName(FString::Printf(TEXT("FRebocapSource%ld"), (long)(FDateTime::UtcNow().ToUnixTimestamp())));
    thread_ = FRunnableThread::Create(this, *ThreadName, 512 * 1024, TPri_Normal);
  }
  return running_;
}

void FRebocapSource::pose_msg_callback(const QuatMsg* msg) {
  if (client_ == nullptr) {
    UE_LOG(LogTemp, Warning, TEXT("pose_msg_callback client is null!!"));
    return;
  }
  if (!running_) {
    return;
  }
  FName subject = "rebocap";
  {
    FLiveLinkStaticDataStruct static_data_struct = FLiveLinkStaticDataStruct(FLiveLinkSkeletonStaticData::StaticStruct());
    FLiveLinkSkeletonStaticData& static_data = *static_data_struct.Cast<FLiveLinkSkeletonStaticData>();

    static TArray<FName> bone_names = {
        rebocap_bones::pelvis,  rebocap_bones::l_hip,      rebocap_bones::r_hip,      rebocap_bones::spine1,   rebocap_bones::l_knee,
        rebocap_bones::r_knee,  rebocap_bones::spine2,     rebocap_bones::l_ankle,    rebocap_bones::r_ankle,  rebocap_bones::spine3,
        rebocap_bones::l_foot,  rebocap_bones::r_foot,     rebocap_bones::neck,       rebocap_bones::l_collar, rebocap_bones::r_collar,
        rebocap_bones::head,    rebocap_bones::l_shoulder, rebocap_bones::r_shoulder, rebocap_bones::l_elbow,  rebocap_bones::r_elbow,
        rebocap_bones::l_wrist, rebocap_bones::r_wrist,    rebocap_bones::l_hand,     rebocap_bones::r_hand,
    };

    static TArray<int32> bone_parents = {
        0,   // 0 -  Pelvis
        0,   // 1 -  L_Hip
        0,   // 2 -  R_Hip
        0,   // 3 -  Spine1
        1,   // 4 -  L_Knee
        2,   // 5 -  R_Knee
        3,   // 6 -  Spine2
        4,   // 7 -  L_Ankle
        5,   // 8 -  R_Ankle
        6,   // 9 -  Spine3
        7,   // 10 - L_Foot
        8,   // 11 - R_Foot
        9,   // 12 - Neck
        9,   // 13 - L_Collar
        9,   // 14 - R_Collar
        12,  // 15 - Head
        13,  // 16 - L_Shoulder
        14,  // 17 - R_Shoulder
        16,  // 18 - L_Elbow
        17,  // 19 - R_Elbow
        18,  // 20 - L_Wrist
        19,  // 21 - R_Wrist
        20,  // 22 - L_Hand
        21,  // 23 - R_Hand
    };
    if (!running_) return;

    static_data.SetBoneNames(bone_names);
    static_data.SetBoneParents(bone_parents);
    client_->PushSubjectStaticData_AnyThread({source_guid_, subject}, ULiveLinkAnimationRole::StaticClass(), MoveTemp(static_data_struct));
  }

  FLiveLinkFrameDataStruct frame = FLiveLinkFrameDataStruct(FLiveLinkAnimationFrameData::StaticStruct());
  FLiveLinkBaseFrameData* base_data = frame.GetBaseData();
  base_data->WorldTime = FLiveLinkWorldTime();
  FLiveLinkAnimationFrameData& frame_data = *frame.Cast<FLiveLinkAnimationFrameData>();
  frame_data.Transforms.Reserve(24);
  for (int i = 0; i < 96; i += 4) {
    FTransform transform;
    transform.SetTranslation({msg->trans[0], msg->trans[1], msg->trans[2]});
    transform.SetRotation(FQuat{msg->quat[i], msg->quat[i + 1], msg->quat[i + 2], msg->quat[i + 3]});
    transform.SetScale3D(FVector::OneVector);
    frame_data.Transforms.Add(MoveTemp(transform));
  }
  client_->PushSubjectFrameData_AnyThread({source_guid_, subject}, MoveTemp(frame));
  // UE_LOG(LogTemp, Warning, TEXT("pose_msg_callback"));
}

FText FRebocapSource::GetSourceType() const { return FText::FromString("rebocap"); }
FText FRebocapSource::GetSourceMachineName() const { return FText::FromString(FString::Printf(TEXT("ws://localhost:%d"), port_)); }
FText FRebocapSource::GetSourceStatus() const { return status_ ? FText::FromString("ok") : FText::FromString("bad"); }

TSharedPtr<FRebocapSource> FRebocapSource::instance_;

TSharedPtr<FRebocapSource> FRebocapSource::GetInstance() { return instance_; }

void FRebocapSource::SetInstance(TSharedPtr<FRebocapSource> instance) { instance_ = std::move(instance); }
